import 'package:flutter_bloc/flutter_bloc.dart';
import 'package:injectable/injectable.dart';

import '../../../../core/usecases/usecase.dart';
import '../../domain/usecases/get_kyc_levels_usecase.dart';
import '../../domain/usecases/get_kyc_applications_usecase.dart';
import '../../domain/usecases/submit_kyc_application_usecase.dart';
import 'kyc_event.dart';
import 'kyc_state.dart';

@injectable
class KycBloc extends Bloc<KycEvent, KycState> {
  final GetKycLevelsUseCase _getKycLevelsUseCase;
  final GetKycApplicationsUseCase _getKycApplicationsUseCase;
  final SubmitKycApplicationUseCase _submitKycApplicationUseCase;

  KycBloc(
    this._getKycLevelsUseCase,
    this._getKycApplicationsUseCase,
    this._submitKycApplicationUseCase,
  ) : super(const KycInitial()) {
    on<KycLevelsLoadRequested>(_onKycLevelsLoadRequested);
    on<KycApplicationsLoadRequested>(_onKycApplicationsLoadRequested);
    on<KycApplicationSubmitted>(_onKycApplicationSubmitted);
    on<KycStateReset>(_onKycStateReset);
    on<KycErrorDismissed>(_onKycErrorDismissed);
  }

  Future<void> _onKycLevelsLoadRequested(
    KycLevelsLoadRequested event,
    Emitter<KycState> emit,
  ) async {
    print('🔵 KYC_BLOC: Loading KYC levels');
    emit(const KycLoading(message: 'Loading KYC levels...'));

    final result = await _getKycLevelsUseCase(NoParams());

    result.fold(
      (failure) {
        print('🔴 KYC_BLOC: Failed to load KYC levels: ${failure.message}');
        emit(KycError(
          failure: failure,
          message: failure.message,
        ));
      },
      (levels) {
        print('🟢 KYC_BLOC: Successfully loaded ${levels.length} KYC levels');
        emit(KycLevelsLoaded(levels: levels));
      },
    );
  }

  Future<void> _onKycApplicationsLoadRequested(
    KycApplicationsLoadRequested event,
    Emitter<KycState> emit,
  ) async {
    print('🔵 KYC_BLOC: Loading KYC applications');
    emit(const KycLoading(message: 'Loading your KYC applications...'));

    final result = await _getKycApplicationsUseCase(NoParams());

    result.fold(
      (failure) {
        print(
            '🔴 KYC_BLOC: Failed to load KYC applications: ${failure.message}');
        emit(KycError(
          failure: failure,
          message: failure.message,
        ));
      },
      (applications) {
        print(
            '🟢 KYC_BLOC: Successfully loaded ${applications.length} KYC applications');
        emit(KycApplicationsLoaded(applications: applications));
      },
    );
  }

  Future<void> _onKycApplicationSubmitted(
    KycApplicationSubmitted event,
    Emitter<KycState> emit,
  ) async {
    print(
        '🔵 KYC_BLOC: Submitting KYC application for level: ${event.levelId}');
    emit(const KycLoading(message: 'Submitting your KYC application...'));

    final params = SubmitKycApplicationParams(
      levelId: event.levelId,
      fields: event.fields,
    );

    final result = await _submitKycApplicationUseCase(params);

    result.fold(
      (failure) {
        print(
            '🔴 KYC_BLOC: Failed to submit KYC application: ${failure.message}');
        emit(KycError(
          failure: failure,
          message: failure.message,
        ));
      },
      (application) {
        print('🟢 KYC_BLOC: Successfully submitted KYC application');
        emit(KycApplicationSubmitSuccess(
          application: application,
          message: 'Your KYC application has been submitted for review',
        ));
      },
    );
  }

  void _onKycStateReset(
    KycStateReset event,
    Emitter<KycState> emit,
  ) {
    print('🔵 KYC_BLOC: Resetting state');
    emit(const KycInitial());
  }

  void _onKycErrorDismissed(
    KycErrorDismissed event,
    Emitter<KycState> emit,
  ) {
    print('🔵 KYC_BLOC: Error dismissed, resetting to initial state');
    emit(const KycInitial());
  }
}
