import 'package:flutter/material.dart';
import 'package:flutter/services.dart';
import 'package:flutter_bloc/flutter_bloc.dart';
import 'dart:io';

import 'injection/injection.dart';
import 'core/config/app_config.dart';
import 'core/constants/api_constants.dart';
import 'core/services/stripe_service.dart';
import 'core/services/websocket_service.dart';
import 'core/services/market_service.dart';
import 'core/services/maintenance_service.dart';
import 'core/theme/app_themes.dart';
import 'core/theme/global_theme_extensions.dart';
import 'core/widgets/app_logo.dart';
import 'core/utils/orientation_helper.dart';
import 'features/auth/presentation/pages/login_page.dart';
import 'features/auth/presentation/bloc/auth_bloc.dart';
import 'features/theme/presentation/bloc/theme_bloc.dart';
import 'features/theme/presentation/bloc/theme_event.dart';
import 'features/theme/presentation/bloc/theme_state.dart';
import 'features/theme/domain/entities/app_theme_entity.dart';
import 'features/home/presentation/pages/home_page.dart';
import 'features/profile/presentation/bloc/profile_bloc.dart';
import 'features/profile/data/services/profile_service.dart';
import 'features/settings/presentation/widgets/settings_provider.dart';
import 'features/settings/presentation/bloc/settings_bloc.dart';
import 'features/settings/presentation/bloc/settings_state.dart';
import 'features/addons/ecommerce/presentation/bloc/cart/cart_bloc.dart';
import 'features/addons/ecommerce/presentation/bloc/wishlist/wishlist_bloc.dart';
import 'core/widgets/maintenance_banner.dart';

void main() async {
  WidgetsFlutterBinding.ensureInitialized();

  // 🔒 GLOBAL ORIENTATION LOCK: Force portrait mode for the entire app
  // Only the fullscreen chart page will manually override this to landscape
  print('🔒 MAIN: Setting global portrait orientation lock');
  await OrientationHelper.lockPortrait();
  print('✅ MAIN: Portrait orientation lock applied globally');

  // Initialize app configuration first
  print('🚀 MAIN: Loading app configuration');
  try {
    await AppConfig.initialize();
    print('✅ MAIN: App configuration loaded successfully');
  } catch (e) {
    print('❌ MAIN: Failed to load app configuration: $e');
    // Show error to user
    runApp(ConfigurationErrorApp(error: e.toString()));
    return;
  }

  print('🚀 MAIN: Initializing dependencies');
  await configureDependencies();

  print('🚀 MAIN: Initializing Stripe');
  final stripeService = getIt<StripeService>();
  try {
    await stripeService.initialize();
  } on SocketException catch (e) {
    debugPrint('⚠️ Stripe initialization failed (offline): $e');
  } catch (e) {
    debugPrint('⚠️ Stripe initialization error: $e');
  }

  print('🚀 MAIN: Initializing global WebSocket and Market services');
  // Initialize global WebSocket service
  final webSocketService = getIt<WebSocketService>();
  try {
    await webSocketService.initializeGlobal();
  } on SocketException catch (e) {
    debugPrint('⚠️ WebSocket init failed (offline): $e');
  } catch (e) {
    debugPrint('⚠️ WebSocket init error: $e');
  }

  // Initialize global Market service
  final marketService = getIt<MarketService>();
  final maintenanceService = getIt<MaintenanceService>();
  try {
    await marketService.initialize();
  } on SocketException catch (e) {
    debugPrint('⚠️ MarketService init failed (offline): $e');
    maintenanceService.setMaintenanceMode(
        true, 'Unable to connect to server. Using offline mode.');
  } catch (e) {
    debugPrint('⚠️ MarketService init error: $e');
    maintenanceService.handleServiceError(e, 'MarketService');
  }

  print('🚀 MAIN: Running app');
  runApp(const CryptoTradingApp());
}

class ConfigurationErrorApp extends StatelessWidget {
  final String error;

  const ConfigurationErrorApp({super.key, required this.error});

  @override
  Widget build(BuildContext context) {
    return MaterialApp(
      home: Scaffold(
        backgroundColor: Colors.grey[900],
        body: Center(
          child: Padding(
            padding: const EdgeInsets.all(24.0),
            child: Column(
              mainAxisAlignment: MainAxisAlignment.center,
              children: [
                const Icon(
                  Icons.error_outline,
                  color: Colors.red,
                  size: 64,
                ),
                const SizedBox(height: 24),
                const Text(
                  'Configuration Error',
                  style: TextStyle(
                    color: Colors.white,
                    fontSize: 24,
                    fontWeight: FontWeight.bold,
                  ),
                ),
                const SizedBox(height: 16),
                Text(
                  'Failed to load app configuration',
                  style: TextStyle(
                    color: Colors.grey[400],
                    fontSize: 16,
                  ),
                ),
                const SizedBox(height: 24),
                Container(
                  padding: const EdgeInsets.all(16),
                  decoration: BoxDecoration(
                    color: Colors.grey[800],
                    borderRadius: BorderRadius.circular(8),
                  ),
                  child: Text(
                    error,
                    style: TextStyle(
                      color: Colors.grey[300],
                      fontSize: 14,
                      fontFamily: 'monospace',
                    ),
                  ),
                ),
                const SizedBox(height: 32),
                Text(
                  'Please ensure assets/config/app_config.json exists and is properly configured.',
                  textAlign: TextAlign.center,
                  style: TextStyle(
                    color: Colors.grey[400],
                    fontSize: 14,
                  ),
                ),
              ],
            ),
          ),
        ),
      ),
    );
  }
}

class CryptoTradingApp extends StatelessWidget {
  const CryptoTradingApp({super.key});

  @override
  Widget build(BuildContext context) {
    print('🚀 MAIN: Building CryptoTradingApp');

    return BlocProvider<ThemeBloc>(
      create: (context) => getIt<ThemeBloc>()..add(const ThemeLoadRequested()),
      child: BlocBuilder<ThemeBloc, ThemeState>(
        builder: (context, themeState) {
          // Determine which theme to use
          ThemeData currentTheme = AppThemes.darkTheme; // Default

          if (themeState is ThemeLoaded) {
            final selectedTheme = themeState.currentTheme;
            final systemTheme = themeState.systemTheme;

            switch (selectedTheme) {
              case AppThemeType.light:
                currentTheme = AppThemes.lightTheme;
                break;
              case AppThemeType.dark:
                currentTheme = AppThemes.darkTheme;
                break;
              case AppThemeType.system:
                // Use system theme
                currentTheme = systemTheme == AppThemeType.dark
                    ? AppThemes.darkTheme
                    : AppThemes.lightTheme;
                break;
            }
          }

          return MaterialApp(
            title: AppConstants.appName,
            theme: currentTheme,
            themeAnimationDuration: const Duration(milliseconds: 300),
            themeAnimationCurve: Curves.easeInOut,
            debugShowCheckedModeBanner: false,
            home: MultiBlocProvider(
              providers: [
                BlocProvider(
                  create: (context) {
                    print(
                        '🚀 MAIN: Creating AuthBloc and checking authentication status');
                    final authBloc = getIt<AuthBloc>();
                    authBloc.add(AuthCheckRequested());
                    return authBloc;
                  },
                ),
                BlocProvider(
                  create: (context) {
                    print('🚀 MAIN: Creating ProfileBloc');
                    final profileBloc = getIt<ProfileBloc>();

                    print(
                        '🚀 MAIN: Initializing ProfileService with ProfileBloc');
                    // Initialize ProfileService with the ProfileBloc
                    final profileService = getIt<ProfileService>();
                    profileService.initialize(profileBloc);
                    print('🚀 MAIN: ProfileService initialization completed');

                    return profileBloc;
                  },
                ),
                BlocProvider(
                  create: (context) {
                    print('🚀 MAIN: Creating CartBloc');
                    return getIt<CartBloc>();
                  },
                ),
                BlocProvider(
                  create: (context) {
                    print('🚀 MAIN: Creating WishlistBloc');
                    return getIt<WishlistBloc>();
                  },
                ),
              ],
              child: const AuthWrapper(),
            ),
          );
        },
      ),
    );
  }
}

class AuthWrapper extends StatelessWidget {
  const AuthWrapper({super.key});

  @override
  Widget build(BuildContext context) {
    print('🚀 MAIN: Building AuthWrapper');

    return BlocListener<AuthBloc, AuthState>(
      listenWhen: (previous, current) {
        print(
            '🚀 MAIN: BlocListener listenWhen - Previous: ${previous.runtimeType}, Current: ${current.runtimeType}');
        return current
            is AuthUnauthenticated; // Only listen to unauthenticated state
      },
      listener: (context, state) {
        print('🚀 MAIN: AuthWrapper received state: ${state.runtimeType}');
        if (state is AuthUnauthenticated) {
          print('🚀 MAIN: User unauthenticated, performing complete cleanup');

          // Perform comprehensive cleanup
          _performLogoutCleanup(context);

          print('🟢 MAIN: Cleanup completed, ready for login');

          // Force a rebuild after cleanup
          WidgetsBinding.instance.addPostFrameCallback((_) {
            print('🚀 MAIN: Forcing rebuild after cleanup');
            // Trigger a rebuild by accessing the bloc again
            if (context.mounted) {
              context.read<AuthBloc>();
            }
          });
        }
      },
      child: BlocListener<ProfileBloc, ProfileState>(
        listener: (context, profileState) {
          // This listener ensures profile auto-fetch happens after ProfileBloc is ready
          final authState = context.read<AuthBloc>().state;
          if (authState is AuthAuthenticated &&
              profileState is ProfileInitial) {
            print('🚀 MAIN: ProfileBloc ready, auto-fetching profile');
            final profileService = getIt<ProfileService>();
            profileService.autoFetchProfile();
          }
        },
        child: BlocBuilder<AuthBloc, AuthState>(
          buildWhen: (previous, current) {
            print(
                '🚀 MAIN: BlocBuilder buildWhen - Previous: ${previous.runtimeType}, Current: ${current.runtimeType}');
            return true; // Always rebuild to ensure state changes are reflected
          },
          builder: (context, state) {
            print('🚀 MAIN: AuthWrapper state: ${state.runtimeType}');

            if (state is AuthLoading) {
              print('🚀 MAIN: Showing loading screen');
              return Scaffold(
                backgroundColor: context.colors.background,
                body: Center(
                  child: Column(
                    mainAxisAlignment: MainAxisAlignment.center,
                    children: [
                      const AnimatedAppLogo(
                        fontSize: 36,
                        showIcon: false,
                        style: LogoStyle.elegant,
                      ),
                      const SizedBox(height: 32),
                      CircularProgressIndicator(
                        color: context.colors.primary,
                        strokeWidth: 3,
                      ),
                      const SizedBox(height: 24),
                      Text(
                        _getLoadingMessage(state),
                        style: context.bodyM.copyWith(
                          color: context.textSecondary,
                        ),
                        textAlign: TextAlign.center,
                      ),
                    ],
                  ),
                ),
              );
            } else if (state is AuthAuthenticated) {
              print(
                  '🚀 MAIN: User authenticated, showing home page with settings check');

              // Trigger profile auto-fetch after the widget tree is built
              WidgetsBinding.instance.addPostFrameCallback((_) {
                print(
                    '🚀 MAIN: Post-frame callback - triggering profile auto-fetch');
                final profileService = getIt<ProfileService>();
                profileService.autoFetchProfile();
              });

              return const SettingsProvider(
                child: SettingsLoadingWrapper(),
              );
            } else if (state is AuthUnauthenticated) {
              print('🚀 MAIN: User unauthenticated, showing login page');
              // Force a frame callback to ensure UI updates
              WidgetsBinding.instance.addPostFrameCallback((_) {
                print(
                    '🚀 MAIN: Post-frame callback after logout - UI should be updated');
              });
              return const MaintenanceAwareScaffold(child: LoginPage());
            } else {
              print('🚀 MAIN: User not authenticated, showing login page');
              return const MaintenanceAwareScaffold(child: LoginPage());
            }
          },
        ),
      ),
    );
  }

  String _getLoadingMessage(AuthState state) {
    // You can enhance this to show different messages based on context
    // For now, we'll show a generic message
    return 'Please wait...';
  }

  void _performLogoutCleanup(BuildContext context) {
    print('🔵 MAIN: Starting comprehensive logout cleanup');

    try {
      // Clear profile service data
      final profileService = getIt<ProfileService>();
      profileService.clearCache();
      profileService.reset();
      print('🟢 MAIN: ProfileService cleaned up');

      // Clear any existing snackbars or dialogs
      ScaffoldMessenger.of(context).clearSnackBars();
      print('🟢 MAIN: UI state cleared');

      // Clear navigation stack by popping all routes except root
      Navigator.of(context).popUntil((route) => route.isFirst);
      print('🟢 MAIN: Navigation stack cleared');

      // Force system UI reset using theme colors
      final isDark = context.isDarkMode;
      SystemChrome.setSystemUIOverlayStyle(
        SystemUiOverlayStyle(
          statusBarColor:
              isDark ? context.colors.background : context.colors.surface,
          statusBarIconBrightness: isDark ? Brightness.light : Brightness.dark,
        ),
      );
      print('🟢 MAIN: System UI reset');

      // Additional cleanup can be added here for other services
      print('🟢 MAIN: Additional service cleanup completed');

      // Force garbage collection
      WidgetsBinding.instance.addPostFrameCallback((_) {
        print('🔵 MAIN: Triggering garbage collection');
      });
    } catch (e) {
      print('🔴 MAIN: Error during logout cleanup: $e');
    }
  }
}

class SettingsLoadingWrapper extends StatelessWidget {
  const SettingsLoadingWrapper({super.key});

  @override
  Widget build(BuildContext context) {
    return BlocBuilder<SettingsBloc, SettingsState>(
      builder: (context, state) {
        // Show loading state until settings are loaded
        if (state is SettingsLoading || state is SettingsInitial) {
          return Scaffold(
            backgroundColor: context.colors.background,
            body: Center(
              child: Column(
                mainAxisAlignment: MainAxisAlignment.center,
                children: [
                  const AppLogo.textOnly(
                    fontSize: 32,
                    style: LogoStyle.elegant,
                  ),
                  const SizedBox(height: 32),
                  CircularProgressIndicator(
                    color: context.colors.primary,
                    strokeWidth: 3,
                  ),
                  const SizedBox(height: 24),
                  Text(
                    'Please wait...',
                    style: context.bodyM.copyWith(
                      color: context.textSecondary,
                    ),
                    textAlign: TextAlign.center,
                  ),
                ],
              ),
            ),
          );
        }

        // Settings loaded successfully, show the HomePage
        return const MaintenanceAwareScaffold(child: HomePage());
      },
    );
  }
}

/// Wrapper widget that shows maintenance banner when needed
class MaintenanceAwareScaffold extends StatelessWidget {
  const MaintenanceAwareScaffold({
    super.key,
    required this.child,
  });

  final Widget child;

  @override
  Widget build(BuildContext context) {
    final maintenanceService = getIt<MaintenanceService>();

    return StreamBuilder<bool>(
      stream: maintenanceService.maintenanceStream,
      initialData: maintenanceService.isInMaintenance,
      builder: (context, snapshot) {
        final isInMaintenance = snapshot.data ?? false;

        if (isInMaintenance) {
          return Scaffold(
            backgroundColor: context.colors.background,
            body: Column(
              children: [
                SafeArea(
                  bottom: false,
                  child: MaintenanceBanner(
                    message: maintenanceService.maintenanceMessage,
                  ),
                ),
                Expanded(child: child),
              ],
            ),
          );
        }

        return child;
      },
    );
  }
}
